<?php
/**
 * ProxStop Class
 * @author tech@dayos.net
 */

class ProxStop
{
	public $apiKey = '';
	public $errors = array();
	public $proxyScoreRisk = array(
		0 => 'No Risk',
		1 => 'Low Risk',
		2 => 'Medium Risk',
		3 => 'High Risk',
		4 => 'Very High Risk',
		5 => 'Extremely High Risk',
	);
	public $telephoneCleansed = array(
		100 => 'No Changes / Exact',
		101 => 'Changed',
		102 => 'Restricted Number',
		103 => 'No Match',
		104 => 'Invalid',
		105 => 'Partial Match',
	);
	public $telephoneTypes = array(
		300 => '(Medium) Undetermined',
		301 => '(Low) Fixed Line',
		302 => '(Medium-Low) Mobile',
		303 => '(Medium-High) PrePaid Mobile',
		304 => '(High) Toll-Free',
		305 => '(High) Non-Fixed VOIP',
		306 => '(High) Pager',
		307 => '(High) Payphone',
		308 => '(High) Invalid',
		309 => '(High) Restricted Number',
		310 => '(Medium-Low) Personal',
		311 => '(Medium-High) Voicemail',
		320 => '(Medium-High) Other',
		400 => '(High) Fraud',
	);
	public $telephoneStatus = array(
		100 => 'Call answered',
		101 => 'Not answered',
		103 => 'Call in progress',
		105 => 'Call not handled yet',
		106 => 'Call failed',
		107 => 'Line busy',
		108 => 'Call canceled',
		129 => 'Call blocked by customer request',
		130	=> 'Call blocked',
		200 => 'Delivered to handset',
		203 => 'Delivered to gateway',
		207 => 'Error delivering SMS to handset (reason unknown)',
		210 => 'Temporary phone error',
		211 => 'Permanent phone error',
		220 => 'Gateway/network cannot route message',
		221 => 'Message expired before delivery',
		222 => 'SMS not supported',
		229 => 'Message blocked by customer request',
		230 => 'Message blocked',
		231 => 'Invalid/unsupported message content',
		250 => 'Final status unknown',
		290 => 'Message in progress',
		291 => 'Queued',
		292 => 'Queued at gateway',
		295 => 'Status delayed',
		500 => 'Transaction not attempted',
		501 => 'Not authorized',
		503 => 'Carrier rejected - temporary problem',
		504 => 'Carrier rejected - permanent error',
		505 => 'Error on gateway - temporary error',
		506 => 'Error on gateway - permanent error',
		507 => 'Invalid destination address',
		508 => 'Invalid source address',
		509 => 'Parameters problem',
		510 => 'Message blocked by subscriber action or request',
		511 => 'Subscriber low on credit',
		512 => 'Roaming error',
		513 => 'Mobile number portability error',
		514 => 'Subscriber absent',
		515 => 'Suspected spam',
		599 => 'Status not available',
	);

	/**
	 * Constructer to initialize this class
	 *
	 * @param array $args	A list of config settings
	 */
	public function __construct($args)
	{
		$this->apiKey = $args['key'];
	}

	/**
	 * Balance
	 */
	public function balance()
	{
		return $this->_callAPI(
			'balance'
		);
	}

	/**
	 * E-mail Lookup
	 *
	 * @param string $email	The E-mail being checked
	 * @param string $ref	A personal note / comment
	 */
	public function emailLookup($email,$ref='')
	{
		return $this->_callAPI(
			'email',
			array(
				'email' => $email,
				'ref' => $ref,
			)
		);
	}

	/**
	 * Proxy Lookup
	 *
	 * @param string $ip	The IP being checked
	 * @param string $ref	A personal note / comment
	 */
	public function proxyLookup($ip,$ref='')
	{
		return $this->_callAPI(
			'ip',
			array(
				'ip' => $ip,
				'ref' => $ref,
			)
		);
	}
	
	/**
	 * Telephone Call
	 *
	 * @param string $country	Country code associated with the phone number
	 * @param string $phone	Phone number, including area code
	 * @param string $code	Verification code, 3-5 digits
	 * @param string $ext	Extension number the system should dial.
	 * @param string $message Language the message is in
	 * @param string $ref	A personal note / comment
	 */
	public function telephoneCall($country,$phone,$code,$ext='',$message='English',$ref='')
	{
		return $this->_callAPI(
			'telephonecall',
			array(
				'country' => $country,
				'phone' => $phone,
				'code' => $code,
				'ext' => $ext,
				'message' => $message,
				'ref' => $ref,
			)
		);
	}
	
	/**
	 * Telephone Lookup
	 *
	 * @param string $country	Country code associated with the phone number
	 * @param string $phone	Phone number, including area code
	 * @param string $ref	A personal note / comment
	 */
	public function telephoneLookup($country,$phone,$ref='')
	{
		return $this->_callAPI(
			'telephonelookup',
			array(
				'country' => $country,
				'phone' => $phone,
				'ref' => $ref,
			)
		);
	}

	/**
	 * Telephone SMS
	 *
	 * @param string $country	Country code associated with the phone number
	 * @param string $phone	Phone number, including area code
	 * @param string $code	Verification code, 3-5 digits
	 * @param string $message	Custom SMS message
	 * @param string $ref	A personal note / comment
	 */
	public function telephoneSMS($country,$phone,$code,$message='',$ref='')
	{
		return $this->_callAPI(
			'telephonesms',
			array(
				'country' => $country,
				'phone' => $phone,
				'code' => $code,
				'message' => $message,
				'ref' => $ref,
			)
		);
	}
	
	/**
	 * Telephone Status
	 *
	 * @param string $refid	Reference ID generated by the ProxStop call or SMS
	 * @param string $code	Verification code, 3-5 digits
	 */
	public function telephoneStatus($refid,$code,$ref='')
	{
		return $this->_callAPI(
			'telephonestatus',
			array(
				'referenceid' => $refid,
				'code' => $code,
			)
		);
	}
	
	/**
	 * Call API URL
	 *
	 * @param string $service	The name of the service being called
	 * @param array $args	A list of arguments being passed to the service
	 */
	protected function _callAPI($service,$args=array())
	{
		// Build URL
		$args['key'] = $this->apiKey;
		$result = trim(@file_get_contents('https://api.proxstop.com/'.$service.'.xml?'.http_build_query($args)));
		
		// Nothing or Error?
		if($result===false || $result=='')
			return false;
			
		// Load XML into PHP
		$xml = simplexml_load_string($result);
		
		// Are there any errors we need to store?
		if(isset($xml->error_code))
		{
			$this->errors = array(
				'code' => $xml->error_code,
				'msg' => $xml->error_msg,
			);
			return false;
		}
		
		// All is good
		return $xml;
	}
}